<?php

namespace WpCafe_Pro\Utils;

defined("ABSPATH") || exit;

class Utilities
{

    use \WpCafe_Pro\Traits\Singleton;

    /**
     * Show Notices
     *
     * @since 1.0.0
     * @return void
     */
    public static function push($notice)
    {

        $defaults = [
            'id'               => '',
            'type'             => 'info',
            'show_if'          => true,
            'message'          => '',
            'class'            => 'wpc-active-notice',
            'dismissible'      => false,
            'btn'              => [],
            'dismissible-meta' => 'user',
            'dismissible-time' => WEEK_IN_SECONDS,
            'data'             => '',
        ];

        $notice = wp_parse_args($notice, $defaults);

        $classes = ['wpc-notice', 'notice'];

        $classes[] = $notice['class'];

        if (isset($notice['type'])) {
            $classes[] = 'notice-' . $notice['type'];
        }

        // Is notice dismissible?
        if (true === $notice['dismissible']) {
            $classes[] = 'is-dismissible';

            // Dismissable time.
            $notice['data'] = ' dismissible-time=' . esc_attr($notice['dismissible-time']) . ' ';
        }

        // Notice ID.
        $notice_id    = 'wpc-sites-notice-id-' . $notice['id'];
        $notice['id'] = $notice_id;

        if (!isset($notice['id'])) {
            $notice_id    = 'wpc-sites-notice-id-' . $notice['id'];
            $notice['id'] = $notice_id;
        } else {
            $notice_id = $notice['id'];
        }

        $notice['classes'] = implode(' ', $classes);

        // User meta.
        $notice['data'] .= ' dismissible-meta=' . esc_attr($notice['dismissible-meta']) . ' ';

        if ('user' === $notice['dismissible-meta']) {
            $expired = get_user_meta(get_current_user_id(), $notice_id, true);
        } elseif ('transient' === $notice['dismissible-meta']) {
            $expired = get_transient($notice_id);
        }

        // Notice visible after transient expire.
        if (isset($notice['show_if'])) {
            if (true === $notice['show_if']) {

                // Is transient expired?
                if (false === $expired || empty($expired)) {
                    self::markup($notice);
                }
            }
        } else {
            self::markup($notice);
        }
    }

    /**
     * return same data function
     *
     * @param $content
     * @return void
     */
    public static function render($content)
    {
        if ($content == "") {
            return "";
        }

        return $content;
    }

    /**
     * Allowed vallid html
     *
     * @param [type] $raw
     * @return void
     */
    public static function kses($raw)
    {
        $allowed_tags = [
            'a'                             => [
                'class'  => [],
                'href'   => [],
                'rel'    => [],
                'title'  => [],
                'target' => [],
            ],
            'input'                         => [
                'value'       => [],
                'type'        => [],
                'size'        => [],
                'name'        => [],
                'checked'     => [],
                'placeholder' => [],
                'id'          => [],
                'class'       => [],
            ],

            'select'                        => [
                'value'       => [],
                'type'        => [],
                'size'        => [],
                'name'        => [],
                'placeholder' => [],
                'id'          => [],
                'class'       => [],
                'option'      => [
                    'value'   => [],
                    'checked' => [],
                ],
            ],

            'textarea'                      => [
                'value'       => [],
                'type'        => [],
                'size'        => [],
                'name'        => [],
                'rows'        => [],
                'cols'        => [],

                'placeholder' => [],
                'id'          => [],
                'class'       => [],
            ],
            'abbr'                          => [
                'title' => [],
            ],
            'b'                             => [],
            'blockquote'                    => [
                'cite' => [],
            ],
            'cite'                          => [
                'title' => [],
            ],
            'code'                          => [],
            'del'                           => [
                'datetime' => [],
                'title'    => [],
            ],
            'dd'                            => [],
            'div'                           => [
                'class' => [],
                'title' => [],
                'style' => [],
            ],
            'dl'                            => [],
            'dt'                            => [],
            'em'                            => [],
            'h1'                            => [
                'class' => [],
            ],
            'h2'                            => [
                'class' => [],
            ],
            'h3'                            => [
                'class' => [],
            ],
            'h4'                            => [
                'class' => [],
            ],
            'h5'                            => [
                'class' => [],
            ],
            'h6'                            => [
                'class' => [],
            ],
            'i'                             => [
                'class' => [],
            ],
            'img'                           => [
                'alt'    => [],
                'class'  => [],
                'height' => [],
                'src'    => [],
                'width'  => [],
            ],
            'li'                            => [
                'class' => [],
            ],
            'ol'                            => [
                'class' => [],
            ],
            'p'                             => [
                'class' => [],
            ],
            'q'                             => [
                'cite'  => [],
                'title' => [],
            ],
            'span'                          => [
                'class' => [],
                'title' => [],
                'style' => [],
            ],
            'iframe'                        => [
                'width'       => [],
                'height'      => [],
                'scrolling'   => [],
                'frameborder' => [],
                'allow'       => [],
                'src'         => [],
            ],
            'strike'                        => [],
            'br'                            => [],
            'strong'                        => [],
            'data-wow-duration'             => [],
            'data-wow-delay'                => [],
            'data-wallpaper-options'        => [],
            'data-stellar-background-ratio' => [],
            'ul'                            => [
                'class' => [],
            ],
            'label'                         => [
                'class' => [],
                'for' => [],
            ]
        ];

        if (function_exists('wp_kses')) { // WP is here
            return wp_kses($raw, $allowed_tags);
        } else {
            return $raw;
        }
    }

    /**
     * Markup Notice.
     *
     * @since 1.0.0
     * @param  array $notice Notice markup.
     * @return void
     */
    public static function markup($notice = [])
    {
?>
        <div id="<?php echo esc_attr($notice['id']); ?>" class="<?php echo esc_attr($notice['classes']); ?>" <?php echo Self::render($notice['data']); ?>>
            <p>
                <?php echo Self::kses($notice['message']); ?>
            </p>
            <?php
            if (!empty($notice['btn'])) : ?>
                <p>
                    <a href="<?php echo esc_url($notice['btn']['url']); ?>" class="button-primary"><?php echo esc_html($notice['btn']['label']); ?></a>
                </p>
            <?php endif; ?>
        </div>
<?php
    }

    /**
     * Check empty and single quote string function
     *
     * @return void
     */
    public static function data_validation_check($data)
    {
        $flag = false;

        if (isset($data) && sanitize_text_field($data) !== '') {
            $flag = true;
        }

        return $flag;
    }

    /**
     * Check empty and single quote string for array function
     *
     * @return void
     */
    public static function data_validation_check_arr($data)
    {
        $flag = false;

        foreach ($data as $key => $value) {

            if (isset($value) && sanitize_text_field($value) !== '') {
                $flag = true;
            }
        }

        return $flag;
    }


    public static function make_wpcafe_ready()
    {
        $basename = 'wp-cafe/wpcafe.php';
        $is_plugin_installed     = self::get_installed_plugin_data($basename);
        $plugin_data             = self::get_plugin_data('wp-cafe', $basename);

        if ($is_plugin_installed) {
            // upgrade plugin - attempt for once
            if (isset($plugin_data->version) && $is_plugin_installed['Version'] != $plugin_data->version) {
                self::upgrade_or_install_plugin($basename);
            }

            // activate plugin
            if (is_plugin_active($basename)) {
                return true;
            } else {
                activate_plugin(self::safe_path(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $basename), '', false, true);
                return true;
            }
        } else {
            // install & activate plugin
            $download_link = isset($plugin_data->download_link) ? $plugin_data->download_link : "";
            if (self::upgrade_or_install_plugin($download_link, false)) {
                return true;
            }
        }
        return false;
    }

    private static function get_plugin_data($slug = '', $basename = '')
    {
        if (empty($slug)) {
            return false;
        }
        $installed_plugin = false;
        if ($basename) {
            $installed_plugin = self::get_installed_plugin_data($basename);
        }

        if ($installed_plugin) {
            return $installed_plugin;
        }

        $args = array(
            'slug' => $slug,
            'fields' => array(
                'version' => false,
            ),
        );

        $response = wp_remote_post(
            'http://api.wordpress.org/plugins/info/1.0/',
            array(
                'body' => array(
                    'action' => 'plugin_information',
                    'request' => serialize((object) $args),
                ),
            )
        );

        if (is_wp_error($response)) {
            return false;
        } else {
            $response = unserialize(wp_remote_retrieve_body($response));

            if ($response) {
                return $response;
            } else {
                return false;
            }
        }
    }

    public static function get_installed_plugin_data($basename = '')
    {
        if (empty($basename)) {
            return false;
        }
        if (!function_exists('get_plugins')) {
            include_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugins = get_plugins();
        return isset($plugins[$basename]) ? $plugins[$basename] : false;
    }

    private static function upgrade_or_install_plugin($basename = '', $upgrade = true)
    {
        if (empty($basename)) {
            return false;
        }
        include_once ABSPATH . 'wp-admin/includes/file.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        include_once ABSPATH . 'wp-admin/includes/class-automatic-upgrader-skin.php';

        $skin = new \Automatic_Upgrader_Skin;
        $upgrader = new \Plugin_Upgrader($skin);
        if ($upgrade == true) {
            $upgrader->upgrade($basename);
        } else {
            $upgrader->install($basename);
            activate_plugin($upgrader->plugin_info(), '', false, true);
        }
        return $skin->result;
    }

    public static function safe_path($path)
    {
        $path = str_replace(['//', '\\\\'], ['/', '\\'], $path);
        return str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path);
    }



    // menu location
    public static function get_menu_location($id = null)
    {
        $menu_category = [];
        try {

            if (is_null($id)) {
                $terms = get_terms([
                    'taxonomy'   => 'wpcafe_location',
                    'hide_empty' => false,
                ]);

                foreach ($terms as $cat) {
                    $menu_category[$cat->term_id] = $cat->name;
                }

                return $menu_category;
            } else {
                // return single menu
                return get_post($id);
            }
        } catch (\Exception $es) {
            return [];
        }
    }
    
}
