<?php
namespace SiteGround_Optimizer\I18n;

use SiteGround_Optimizer\Helper\Helper;

use CharlesRumley\PoToJson;

/**
 * I18n functions.
 */
class I18n {

	/**
	 * Create a new helper.
	 */
	public function __construct() {
		// Load the plugin textdomain.
		add_action( 'after_setup_theme', array( $this, 'load_textdomain' ), 9999 );
		// Generate JSON translations.
		add_action( 'upgrader_process_complete', array( $this, 'update_json_translations' ), 10, 2 );
	}

	/**
	 * Load the plugin textdomain.
	 *
	 * @since 1.0.0
	 */
	public function load_textdomain() {
		load_plugin_textdomain(
			'sg-cachepress',
			false,
			'sg-cachepress/languages'
		);
	}

	/**
	 * Update json translations on translations update.
	 *
	 * @since  5.5.5
	 *
	 * @param  {WP_Upgrader} $upgrader WP_Upgrader instance.
	 * @param  array         $extra    Array of bulk item update data.
	 */
	public function update_json_translations( $upgrader, $extra ) {
		// Bail if we don't update the translations.
		if (
			'update' !== $extra['action'] &&
			'translation' !== $extra['type']
		) {
			return;
		}

		// Bail if there are no translations.
		if ( empty( $extra['translations'] ) ) {
			return;
		}

		// Check for SG Optimizer translations.
		$keys = array_keys( array_column( $extra['translations'], 'slug' ), 'sg-cachepress' );

		// Bail if there are no plugin translations.
		if ( empty( $keys ) ) {
			return;
		}

		// Setup the WP Filesystem.
		$wp_filesystem = Helper::setup_wp_filesystem();

		// Init the convertor class.
		$po_to_json = new PoToJson();

		foreach ( $keys as $key ) {
			// Convert a PO file to Jed-compatible JSON.
			$json = $po_to_json
						->withPoFile( WP_CONTENT_DIR . '/languages/plugins/sg-cachepress-' . $extra['translations'][ $key ]['language'] . '.po' )
						->toJedJson( false, 'sg-cachepress' );

			// Convert and get the json content.
			$content = json_decode( $json, true );

			// Build the json filepath.
			$json_filepath = WP_CONTENT_DIR . '/languages/plugins/sg-cachepress-' . $extra['translations'][ $key ]['language'] . '.json';

			// Create the file if donesn't exists.
			if ( ! is_file( $json_filepath ) ) {
				// Create the new file.
				$wp_filesystem->touch( $json_filepath );
			}

			// Add the translations to the file.
			$wp_filesystem->put_contents(
				$json_filepath,
				json_encode( $content['locale_data']['sg-cachepress'] )
			);
		}
	}

	/**
	 * Get i18n strings as a JSON-encoded string
	 *
	 * @since 5.0.0
	 *
	 * @return string The locale as JSON
	 */
	public static function get_i18n_data_json() {
		// Get the user locale.
		$locale = get_user_locale();

		// Possible langugaes paths.
		$dirs = array(
			'wp-content/languages/plugins/',
			'wp-content/plugins/sg-cachepress/languages/json/',
		);

		foreach ( $dirs as $dir ) {
			// Build the full path to the file.
			$i18n_json = ABSPATH . $dir . 'sg-cachepress-' . $locale . '.json';

			// Check if the files exists and it's readable.
			if ( is_file( $i18n_json ) && is_readable( $i18n_json ) ) {
				// Get the locale data.
				$locale_data = @file_get_contents( $i18n_json );
				if ( $locale_data ) {
					return $locale_data;
				}
			}
		}


		// Return valid empty Jed locale.
		return json_encode(
			array(
				'' => array(
					'domain' => 'sg-cachepress',
					'lang'   => is_admin() ? get_user_locale() : get_locale(),
				),
			)
		);
	}
}
