<?php
namespace WpCafe_Pro;

defined( 'ABSPATH' ) || exit;

use WpCafe_Pro\Autoloader;
use WpCafe_Pro\Core;
use WpCafe_Pro\Utils\Utilities;
use WpCafe_Pro\Widgets\Manifest;

/**
 * Autoload all classes
 */
require_once plugin_dir_path( __FILE__ ) . '/autoloader.php';

final class Plugin {

    const version = '1.1.7';

    private static $instance;
    /**
     * Register action
     */
    private function __construct() {
        // load autoload method
        Autoloader::run();
    }

    /**
     * Main function
     *
     * @return void
     */
    public function init() {
        $this->prepare_wpcafe();

        // activation and deactivation hook
        register_activation_hook( __FILE__, [$this, 'active'] );
        register_deactivation_hook( __FILE__, [$this, 'deactive'] );

        //enqueue file
        \WpCafe_Pro\Core\Enqueue\Enqueue::instance()->init();

        // define constant
        $this->define_constants();

        // fire in every plugin load action
        $this->init_plugin();
    }

    /**
     * Prepare wp-cafe free version if not activated
     *
     * @return void
     */
    private function prepare_wpcafe(){

        // if wpcafe not installed
        if ( !did_action( 'wpcafe/after_load' ) ) {
            if ( \WpCafe_Pro\Utils\Utilities::make_wpcafe_ready() ) {
                // redirect to plugin dashboard
                wp_safe_redirect( "admin.php?page=cafe_menu" );
            };
        }
    }

    /**
     * do stuff on active
     *
     * @return void
     */
    public function active() {
        $installed = get_option( 'wpc_pro_cafe_installed' );

        if ( !$installed ) {
            update_option( 'wpc_pro_cafe_installed', time() );
        }

        update_option( 'wpc_pro_cafe_version', WPC_PRO_VERSION );
    }

    /**
     * do stuff on deactive
     *
     * @return void
     */
    public function deactive() {
        flush_rewrite_rules();
    }

    /**
     * Define all constance
     *
     * @return void
     */
    public function define_constants() {
        define( 'WPC_PRO_VERSION', self::version . time() );
        define( 'WPC_PRO_PATH', plugin_dir_url( __FILE__ ) );
        define( 'WPC_PRO_DIR', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
        define( 'WPC_PRO_CORE', WPC_PRO_DIR."/core/");
        define( 'WPC_PRO_ASSETS', WPC_PRO_PATH . 'assets/' );
    }

    public static function instance() {

        if ( !self::$instance ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Load all class
     *
     * @return void
     */
    public function init_plugin() {
        
        if ( !did_action( 'wpcafe/after_load' ) ) {
            add_action( 'admin_notices', [$this, 'notice_wpcafe_not_active'] );
            return;
        }

        // call ajax submit
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            \WpCafe_Pro\Core\Action\Ajax_Action::instance()->init();
        }

        // load text domain
        load_plugin_textdomain( 'wpcafe-pro' );

        // load elementor
        Manifest::instance()->init();

        // call admin action
        if ( is_admin() ) {
            Core::instance()->admin_action_init();
        }

        // load shortcode
        $this->register_widgets();
        
        //fire-up all woocommerce related hooks
        if( file_exists( WPC_PRO_DIR . '/core/woocommerce/hooks.php' )){
            include_once WPC_PRO_DIR . '/core/woocommerce/hooks.php';
        }

    }

    /**
     * Load on plugin
     *
     * @return void
     */
    public function notice_wpcafe_not_active() {

        if ( isset( $_GET['activate'] ) ) {
            unset( $_GET['activate'] );
        }

        if ( file_exists( WP_PLUGIN_DIR . '/wp-cafe/wpcafe.php' ) ) {
            $btn['label'] = esc_html__( 'Activate WpCafe', 'wpcafe-pro' );
            $btn['url']   = wp_nonce_url( 'plugins.php?action=activate&plugin=wp-cafe/wpcafe.php&plugin_status=all&paged=1', 'activate-plugin_wp-cafe/wpcafe.php' );
        } else {
            $btn['label'] = esc_html__( 'Install WpCafe', 'wpcafe-pro' );
            $btn['url']   = wp_nonce_url( self_admin_url( 'update.php?action=install-plugin&plugin=wp-cafe' ), 'install-plugin_wp-cafe' );
        }

        Utilities::push(
            [
                'id'          => 'unsupported-wpcafe-version',
                'type'        => 'error',
                'dismissible' => true,
                'btn'         => $btn,
                'message'     => sprintf( esc_html__( 'WpCafe Pro requires WpCafe  , which is currently NOT RUNNING.', 'wpcafe-pro' ) ),
            ]
        );
    }

    /**
     * Register shortcode function
     *
     * @return void
     */
    public function register_widgets() {
        \WpCafe_Pro\Core\Widgets\Hook::instance()->init();
    }

}
